/* Uno_sketch_for_E32_TTL_100_LoRa_module
 *
 * This sketch is to allow an Arduino Uno to interface with an Ebyte
 * E32-TTL-100 LoRa transceiver module, based on a Semtech SX-1278 IC.
 * This module includes a small MCU (an STM8L151C2) which acts as a
 * controller and a UART-SPI bridge. As a result we are able to 
 * communicate with the module via a standard serial port.
 * The only complication is that Arduino UNO modules only have a single 
 * hardware UART capable of performing serial data communication, using 
 * digital I/O pins 0 (Rx) and 1 (Tx). This port is normally connected
 * to the board's USB-serial interface, used for uploading sketches to
 * the Uno and also for communicating with the PC via the IDE's Serial
 *  Monitor. Any other serial ports which may be needed by a sketch must
 * be 'software serial' ports, using functions in the 'SoftwareSerial'
 * library built into the Arduino IDE. That's what we use here to 
 * communicate with the E32-TTL-100 module (at 9600bps).
 * 
 * The connections between the seven pins of the E32-TTL-100 module and
 * the pins of the Arduino Uno are assumed by this sketch to be:
 * E32-TTL-100          Arduino Uno
 * M0     <-----------> D7 (IO7)
 * M1     <-----------> D8 (IO8)
 * RXD    <-----------> D11 (IO11) NOTE: use 4.7k ohm series resistor
 * TXD    <-----------> D10 (IO10) NOTE: use 4.7k ohm series resistor
 * AUX    <-----------> A0  (ADC0)
 * VCC    <-----------> 5V  (via a 3.3V regulator)
 * GND    <-----------> GND
 * 
 * Note that before using the E32-TTL-100 LoRa module in 'normal' 
 * (transparent comms) mode, it should hsve an antenna connected to the 
 * RF output socket. Otherwise the module may be damaged.
 * 
 * Note also that for legal use in Australia, the E32-TTL-100 module 
 * should be programmed to operate at a centre frequency of 434MHz, and 
 * with an RF output power of 25mW or +14dBm. This sketch does so.
 * 
 * Written by Jim Rowe for Silicon Chip
 * Last revised 20/12/2018 at 3:30 pm
 */
#include <SoftwareSerial.h>

// declare array to store module set parameters
byte SetParam[6] = {0xC0,0x00,0x00,0x1C,0x18,0x46};
/* 0xC0 = command code for setting module parameters
 * 0x00 = high byte of address for LoRa module
 * 0x00 = low byte of address for LoRa module
 * 0x1C = 8N1, UART rate 9600 baud, air rate 9600 baud
 * 0x18 = desired frequency (410 + 18h/24d = 434MHz)
 * 0x46 = transp Tx, wakeup in 250ms, FEC on & 25mW output
 */
const byte ReqPar = 0xC1;  // cmd code to request saved parameters
const byte ReqFWV = 0xC3;  // cmd code to request FW version
byte RetByte = 0x00;       // byte used to save returned data
byte junkByt = 0x00;       // byte for saving unwanted 'echoes'
int x = 0;                 // loop counter

SoftwareSerial mySerial(10,11); // set IO10 pin as RX pin, IO11 pin as TX pin
  
void setup()
{
  pinMode(7, OUTPUT);   // set pins IO7 and IO8 as outputs (for M0 & M1)
  pinMode(8, OUTPUT);
  pinMode(A0, INPUT);  // also set pin A0 as a digital input (for AUX)
  
  // then open hardware serial port and wait for port to open
  Serial.begin(9600);
  while(!Serial)
  {
    // wait for port to open
  }
  Serial.println("Uno sketch for E12-TTL-100 LoRa now running");
  
  mySerial.begin(9600);   // now open software serial port
  while(!mySerial)
  {
    // and wait for port to open
  }
  Serial.println("Software serial port also open");
  
  // next set pins D7 (M0) and D8 (M1) high, to set E32-TTL-100 module
  // into sleep/parameter setting mode (Mode 3)
  digitalWrite(7, HIGH);
  digitalWrite(8, HIGH);
  delay(5);   // delay 5ms to allow module to make mode change
  Serial.println("Module's M0 & M1 pins now set to 1 (Mode 3)");
  
  // then send the parameter string to prepare the module for operation
  // at a centre frequency of 434MHz, with an output power of 25mW
  for(x = 0; x < 6; x++)
  {
    mySerial.write(SetParam[x]);  // send the bytes, in order
  }
  Serial.println("Module initialised");
  Serial.println(" ");
  
  Serial.print("Saved parameters = ");
  // next send a request for the saved parameters
  for (x = 0; x < 3; x++)
  {
    mySerial.write(ReqPar); // send 0xC1 three times
  }
  // then turn on the software serial port for reception
  mySerial.listen();
  // then listen for the returned bytes
  if(mySerial.available())
  { 
    for (x = 0; x < 6; x++)
    {
      RetByte = byte(mySerial.read());  // read a byte
      Serial.print(RetByte, HEX); // & print it out      
      if(RetByte == 0)  // but if it's a zero
      {
        Serial.print(RetByte, HEX); //print it again
      }
    }
    Serial.println(" ");    // then end the print line
  }
  // next read & discard the echo of saved parameters
  if(mySerial.available())
  {  
    for (x = 0; x < 6; x++)
    {
      junkByt = mySerial.read();
    }
  }
  
  Serial.print("Firmware version = "); 
  // next send a request for the module's firmware version
  for (x = 0; x < 3; x++)
  {
    mySerial.write(ReqFWV); // send 0xC3 three times
  }
  delay(20);
  // and listen for the returned bytes
  if(mySerial.available())
  {  
    for (x = 0; x < 4; x++)
    {
      RetByte = mySerial.read();  // read a byte
      Serial.print(RetByte, HEX); // & print it out
    }
  Serial.println(" ");    // then end the print line
  }
  // initialisation sequence done, so we can now go to Mode 0
  digitalWrite(7, LOW);
  digitalWrite(8, LOW);
  delay(5);     // wait 5ms for module to change mode
  Serial.println("Module now in Mode 0. Ready to transmit & receive");
}
//  now we can enter the main program loop, to relay data each way
void loop()
{
  if (Serial.available())
  {                                 // if anything comes in USB serial from PC,
    mySerial.write(Serial.read());  // read it and send it out via software serial
  }
  if (mySerial.available())
  {                                 // or if anything comes in from LoRa module,
    Serial.write(mySerial.read());   // read it and send it out via USB serial
  }
}
